using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Text;

/// <summary>
/// NovaGodina - Finding e-similar spanning tree (Optimized Version)
/// 
/// Key optimization: Manhattan MST can be computed in O(n log n) using sweep line!
/// 
/// For Manhattan distance MST, each point connects to at most 8 neighbors
/// (one in each octant). This reduces the complete graph to O(n) edges.
/// 
/// Algorithm for Manhattan MST:
/// 1. Transform coordinates to find candidates in 4 regions (then rotate for others)
/// 2. For each point, find nearest neighbor in each octant using BIT/sweep line
/// 3. This gives O(n) candidate edges
/// 4. Run Kruskal's on these O(n) edges: O(n log n)
/// </summary>
class NovaGodina
{
    static int n;
    static double epsilon;
    static double[] px, py;
    
    static StreamReader reader;
    static StringBuilder output;
    
    static double ManhattanDist(int i, int j)
    {
        return Math.Abs(px[i] - px[j]) + Math.Abs(py[i] - py[j]);
    }
    
    // Union-Find for Kruskal's algorithm
    static int[] ufParent, ufRank;
    
    static int Find(int x)
    {
        if (ufParent[x] != x)
            ufParent[x] = Find(ufParent[x]);
        return ufParent[x];
    }
    
    static bool Union(int x, int y)
    {
        int rx = Find(x), ry = Find(y);
        if (rx == ry) return false;
        if (ufRank[rx] < ufRank[ry]) { int t = rx; rx = ry; ry = t; }
        ufParent[ry] = rx;
        if (ufRank[rx] == ufRank[ry]) ufRank[rx]++;
        return true;
    }
    
    // Fenwick Tree (BIT) for minimum query
    static int[] bitVal;
    static int[] bitIdx;
    static int bitSize;
    
    static void BitInit(int sz)
    {
        bitSize = sz + 2;
        bitVal = new int[bitSize];
        bitIdx = new int[bitSize];
        for (int i = 0; i < bitSize; i++)
        {
            bitVal[i] = int.MaxValue;
            bitIdx[i] = -1;
        }
    }
    
    static void BitUpdate(int pos, int val, int idx)
    {
        pos++; // 1-indexed
        while (pos > 0)
        {
            if (val < bitVal[pos])
            {
                bitVal[pos] = val;
                bitIdx[pos] = idx;
            }
            pos -= pos & (-pos);
        }
    }
    
    static int BitQuery(int pos)
    {
        pos++; // 1-indexed
        int minVal = int.MaxValue;
        int minIdx = -1;
        while (pos < bitSize)
        {
            if (bitVal[pos] < minVal)
            {
                minVal = bitVal[pos];
                minIdx = bitIdx[pos];
            }
            pos += pos & (-pos);
        }
        return minIdx;
    }
    
    /// <summary>
    /// Compute Manhattan MST in O(n log n)
    /// Returns adjacency list representation
    /// </summary>
    static List<int>[] ComputeManhattanMST()
    {
        // Candidate edges: for each point, find nearest in each of 4 regions
        // (y >= x, y > -x, etc.) then get 4 more by coordinate flips
        
        List<(double dist, int u, int v)> edges = new List<(double, int, int)>();
        
        // We'll work with integer coordinates (scaled)
        // For floating point, we use coordinate compression
        
        // Point indices
        int[] idx = new int[n];
        for (int i = 0; i < n; i++) idx[i] = i;
        
        // For each of 4 transformations:
        // 1. (x, y)
        // 2. (y, x)
        // 3. (-x, y)
        // 4. (y, -x)
        // In each, find points in region y-x >= 0, y+x > 0 (octant)
        
        double[] tx = new double[n];
        double[] ty = new double[n];
        
        for (int transform = 0; transform < 4; transform++)
        {
            // Apply transformation
            for (int i = 0; i < n; i++)
            {
                double ox = px[i], oy = py[i];
                switch (transform)
                {
                    case 0: tx[i] = ox; ty[i] = oy; break;
                    case 1: tx[i] = oy; ty[i] = ox; break;
                    case 2: tx[i] = -ox; ty[i] = oy; break;
                    case 3: tx[i] = oy; ty[i] = -ox; break;
                }
            }
            
            // Coordinate compress y-x values
            double[] ymx = new double[n];
            for (int i = 0; i < n; i++) ymx[i] = ty[i] - tx[i];
            double[] sortedYmx = (double[])ymx.Clone();
            Array.Sort(sortedYmx);
            
            Dictionary<double, int> ymxRank = new Dictionary<double, int>();
            int rank = 0;
            for (int i = 0; i < n; i++)
            {
                if (!ymxRank.ContainsKey(sortedYmx[i]))
                    ymxRank[sortedYmx[i]] = rank++;
            }
            
            // Sort points by x+y decreasing, then process
            Array.Sort(idx, (a, b) => (ty[b] + tx[b]).CompareTo(ty[a] + tx[a]));
            
            BitInit(rank);
            
            for (int i = 0; i < n; i++)
            {
                int u = idx[i];
                int ymxR = ymxRank[ymx[u]];
                
                // Query for nearest point with y-x >= current y-x
                int nearest = BitQuery(ymxR);
                
                if (nearest != -1)
                {
                    double dist = ManhattanDist(u, nearest);
                    edges.Add((dist, Math.Min(u, nearest), Math.Max(u, nearest)));
                }
                
                // Update BIT with current point: key = y-x, value = y+x
                int val = (int)((ty[u] + tx[u]) * 1e9); // Scale for integer BIT
                BitUpdate(ymxR, val, u);
            }
            
            // Reset idx for next transform
            for (int i = 0; i < n; i++) idx[i] = i;
        }
        
        // Remove duplicates and sort by distance
        var edgeSet = new HashSet<(int, int)>();
        var uniqueEdges = new List<(double dist, int u, int v)>();
        
        foreach (var e in edges)
        {
            var key = (e.u, e.v);
            if (!edgeSet.Contains(key))
            {
                edgeSet.Add(key);
                uniqueEdges.Add(e);
            }
        }
        
        uniqueEdges.Sort((a, b) => a.dist.CompareTo(b.dist));
        
        // Kruskal's algorithm
        ufParent = new int[n];
        ufRank = new int[n];
        for (int i = 0; i < n; i++) { ufParent[i] = i; ufRank[i] = 0; }
        
        List<int>[] mstAdj = new List<int>[n];
        for (int i = 0; i < n; i++) mstAdj[i] = new List<int>();
        
        int edgeCount = 0;
        foreach (var e in uniqueEdges)
        {
            if (Union(e.u, e.v))
            {
                mstAdj[e.u].Add(e.v);
                mstAdj[e.v].Add(e.u);
                edgeCount++;
                if (edgeCount == n - 1) break;
            }
        }
        
        return mstAdj;
    }
    
    /// <summary>
    /// Fallback: Simple O(n²) Prim's for small n or when optimized version has issues
    /// </summary>
    static List<int>[] ComputeMSTSimple()
    {
        var mstAdj = new List<int>[n];
        for (int i = 0; i < n; i++) mstAdj[i] = new List<int>();
        
        bool[] inMST = new bool[n];
        double[] minDist = new double[n];
        int[] parent = new int[n];
        
        for (int i = 0; i < n; i++)
        {
            minDist[i] = double.MaxValue;
            parent[i] = -1;
        }
        minDist[0] = 0;
        
        for (int iter = 0; iter < n; iter++)
        {
            int u = -1;
            double best = double.MaxValue;
            for (int i = 0; i < n; i++)
            {
                if (!inMST[i] && minDist[i] < best)
                {
                    best = minDist[i];
                    u = i;
                }
            }
            
            if (u == -1) break;
            inMST[u] = true;
            
            if (parent[u] != -1)
            {
                mstAdj[u].Add(parent[u]);
                mstAdj[parent[u]].Add(u);
            }
            
            for (int v = 0; v < n; v++)
            {
                if (!inMST[v])
                {
                    double d = ManhattanDist(u, v);
                    if (d < minDist[v])
                    {
                        minDist[v] = d;
                        parent[v] = u;
                    }
                }
            }
        }
        
        return mstAdj;
    }
    
    static double[] ReadDoubles()
    {
        string line = reader.ReadLine();
        string[] parts = line.Split(new char[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);
        double[] result = new double[parts.Length];
        for (int i = 0; i < parts.Length; i++)
            result[i] = double.Parse(parts[i], CultureInfo.InvariantCulture);
        return result;
    }
    
    static void Main()
    {
        reader = new StreamReader(Console.OpenStandardInput(), Encoding.ASCII, false, 1 << 16);
        output = new StringBuilder();
        
        double[] firstLine = ReadDoubles();
        n = (int)firstLine[0];
        epsilon = firstLine[1];
        
        px = new double[n];
        py = new double[n];
        
        for (int i = 0; i < n; i++)
        {
            double[] coords = ReadDoubles();
            px[i] = coords[0];
            py[i] = coords[1];
        }
        
        if (n == 1)
        {
            Console.WriteLine("1\n1");
            return;
        }
        
        if (epsilon >= 2.0)
        {
            Console.WriteLine("1\n1");
            return;
        }
        
        // Use optimized MST for large n, fallback for small n
        List<int>[] mstAdj;
        if (n <= 2000)
            mstAdj = ComputeMSTSimple();
        else
            mstAdj = ComputeManhattanMST();
        
        double sampleThreshold = epsilon * 0.49;
        
        // Root at 0, build tree structure
        int[] parent = new int[n];
        List<int>[] children = new List<int>[n];
        for (int i = 0; i < n; i++) { parent[i] = -1; children[i] = new List<int>(); }
        
        bool[] visited = new bool[n];
        Queue<int> bfs = new Queue<int>();
        bfs.Enqueue(0);
        visited[0] = true;
        
        while (bfs.Count > 0)
        {
            int u = bfs.Dequeue();
            foreach (int v in mstAdj[u])
            {
                if (!visited[v])
                {
                    visited[v] = true;
                    parent[v] = u;
                    children[u].Add(v);
                    bfs.Enqueue(v);
                }
            }
        }
        
        // DFS to select vertices
        List<int> Q = new List<int>();
        List<(int, int)> T = new List<(int, int)>();
        
        int[] lastSel = new int[n];
        double[] distFromLast = new double[n];
        
        Q.Add(0);
        lastSel[0] = 0;
        distFromLast[0] = 0;
        
        Stack<int> stk = new Stack<int>();
        int[] nxtCh = new int[n];
        stk.Push(0);
        
        while (stk.Count > 0)
        {
            int u = stk.Peek();
            if (nxtCh[u] < children[u].Count)
            {
                int v = children[u][nxtCh[u]++];
                double ed = ManhattanDist(u, v);
                double cd = distFromLast[u] + ed;
                
                bool sel = children[v].Count == 0 || children[v].Count >= 2 || cd >= sampleThreshold || ed >= sampleThreshold;
                
                if (sel)
                {
                    Q.Add(v);
                    T.Add((lastSel[u], v));
                    lastSel[v] = v;
                    distFromLast[v] = 0;
                }
                else
                {
                    lastSel[v] = lastSel[u];
                    distFromLast[v] = cd;
                }
                stk.Push(v);
            }
            else stk.Pop();
        }
        
        // Output
        Dictionary<int, int> qPos = new Dictionary<int, int>();
        for (int i = 0; i < Q.Count; i++) qPos[Q[i]] = i + 1;
        
        output.AppendLine(Q.Count.ToString());
        for (int i = 0; i < Q.Count; i++)
        {
            if (i > 0) output.Append(' ');
            output.Append(Q[i] + 1);
        }
        output.AppendLine();
        
        foreach (var (f, t) in T)
            output.AppendLine($"{qPos[f]} {qPos[t]}");
        
        Console.Write(output);
    }
}